<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../../error.log');
error_reporting(E_ALL);

header('Content-Type: application/json');

session_start();

// Check authentication
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'author') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit();
}

require_once __DIR__ . '/../../common/journal_config_new.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

try {
    $author_id = $_SESSION['user_id'];
    $action = $_POST['action'] ?? 'submit';
    $status = ($action === 'draft') ? 'draft' : 'submitted';
    
    // Validate required fields for submission (not draft)
    if ($action === 'submit') {
        $required_fields = ['journal_id', 'desired_issue', 'title', 'abstract', 'keywords', 'total_authors'];
        foreach ($required_fields as $field) {
            if (empty($_POST[$field])) {
                throw new Exception("Field '$field' is required.");
            }
        }
    }
    
    // Handle file upload
    $file_url = null;
    if (isset($_FILES['manuscript_file']) && $_FILES['manuscript_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['manuscript_file'];
        $allowed_extensions = ['doc', 'docx', 'rtf'];
        $max_size = 2 * 1024 * 1024; // 2 MB
        
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_extensions)) {
            throw new Exception('Invalid file type. Only .doc, .docx, and .rtf files are allowed.');
        }
        
        if ($file['size'] > $max_size) {
            throw new Exception('File size exceeds 2 MB limit.');
        }
        
        $upload_dir = __DIR__ . '/uploads/manuscripts/';
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        $unique_filename = uniqid('manuscript_', true) . '.' . $file_extension;
        $file_path = $upload_dir . $unique_filename;
        
        if (move_uploaded_file($file['tmp_name'], $file_path)) {
            $file_url = 'uploads/manuscripts/' . $unique_filename;
        } else {
            throw new Exception('Failed to upload file.');
        }
    } else if ($action === 'submit') {
        throw new Exception('Manuscript file is required for submission.');
    }
    
    // Collect author information
    $authors = [];
    $author_names = $_POST['author_name'] ?? [];
    $author_affiliations = $_POST['author_affiliation'] ?? [];
    
    for ($i = 0; $i < count($author_names); $i++) {
        if (!empty($author_names[$i])) {
            $authors[] = [
                'name' => trim($author_names[$i]),
                'affiliation' => trim($author_affiliations[$i] ?? '')
            ];
        }
    }
    
    // Collect corresponding author information
    $corresponding_author = [
        'name' => trim($_POST['corresponding_author_name'] ?? ''),
        'email' => trim($_POST['corresponding_author_email'] ?? ''),
        'mobile' => trim($_POST['corresponding_author_mobile'] ?? ''),
        'whatsapp' => trim($_POST['corresponding_author_whatsapp'] ?? ''),
        'department' => trim($_POST['corresponding_author_department'] ?? ''),
        'organisation' => trim($_POST['corresponding_author_organisation'] ?? ''),
        'city' => trim($_POST['corresponding_author_city'] ?? ''),
        'state' => trim($_POST['corresponding_author_state'] ?? ''),
        'country' => trim($_POST['corresponding_author_country'] ?? ''),
        'type' => trim($_POST['author_type'] ?? ''),
        'category' => trim($_POST['author_category'] ?? '')
    ];
    
    // Generate tracking code
    $tracking_code = 'IJRSI-' . date('Y') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    
    // Prepare insert query
    $insert_query = "INSERT INTO submissions (
        journal_id,
        author_id,
        desired_issue,
        title,
        abstract,
        keywords,
        total_authors,
        num_pages,
        authors,
        corresponding_author,
        file_url,
        status,
        fee_paid,
        tracking_code,
        url
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $pdo->prepare($insert_query);
    
    if (!$stmt) {
        throw new Exception('Failed to prepare statement: ' . $pdo->errorInfo()[2]);
    }
    
    // Bind parameters
    $journal_id = intval($_POST['journal_id'] ?? 1);
    $desired_issue = trim($_POST['desired_issue'] ?? '');
    $title = trim($_POST['title'] ?? '');
    $abstract = trim($_POST['abstract'] ?? '');
    $keywords = trim($_POST['keywords'] ?? '');
    $total_authors = intval($_POST['total_authors'] ?? 1);
    $num_pages = intval($_POST['num_pages'] ?? 0);
    $authors_json = json_encode($authors);
    $corresponding_author_json = json_encode($corresponding_author);
    $fee_paid = 0;
    $url = $_SERVER['HTTP_REFERER'] ?? null;
    
    $stmt->execute([
        $journal_id,
        $author_id,
        $desired_issue,
        $title,
        $abstract,
        $keywords,
        $total_authors,
        $num_pages,
        $authors_json,
        $corresponding_author_json,
        $file_url,
        $status,
        $fee_paid,
        $tracking_code,
        $url
    ]);
    
    $submission_id = $pdo->lastInsertId();
    
    // Send email notification for submissions (not drafts)
    if ($action === 'submit' && !empty($corresponding_author['email'])) {
        $to = $corresponding_author['email'];
        $subject = "Manuscript Submission Confirmation - RSIS International";
        
        $message = "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #003366; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f9f9f9; }
                .tracking-code { background: white; padding: 15px; margin: 20px 0; border-left: 4px solid #009688; font-size: 18px; font-weight: bold; }
                .button { display: inline-block; padding: 12px 30px; background: #003366; color: white; text-decoration: none; border-radius: 4px; margin: 20px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h2>Manuscript Submission Confirmation</h2>
                </div>
                <div class='content'>
                    <p>Dear " . htmlspecialchars($corresponding_author['name']) . ",</p>
                    
                    <p>Thank you for submitting your manuscript to RSIS International.</p>
                    
                    <p><strong>Manuscript Title:</strong> " . htmlspecialchars($title) . "</p>
                    
                    <div class='tracking-code'>
                        Tracking Code: " . htmlspecialchars($tracking_code) . "
                    </div>
                    
                    <p>Your manuscript has been successfully received and is now under review. You can track the status of your submission using your tracking code.</p>
                    
                    <p>You can log in to your author dashboard to:</p>
                    <ul>
                        <li>View the status of your submission</li>
                        <li>Upload revised manuscripts if requested</li>
                        <li>Communicate with the editorial team</li>
                        <li>Download correspondence and reports</li>
                    </ul>
                    
                    <a href='https://rsisinternational.org/journals/common/author/author_dashboard.php' class='button'>Access Author Dashboard</a>
                    
                    <p>If you have any questions, please don't hesitate to contact us.</p>
                    
                    <p>Best regards,<br>
                    RSIS International Editorial Team</p>
                </div>
                <div class='footer'>
                    <p>&copy; " . date('Y') . " RSIS International. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: RSIS International <noreply@rsisinternational.org>" . "\r\n";
        $headers .= "Reply-To: support@rsisinternational.org" . "\r\n";
        
        mail($to, $subject, $message, $headers);
    }
    
    // Return success response
    if ($action === 'draft') {
        echo json_encode([
            'success' => true,
            'message' => 'Manuscript saved as draft successfully!',
            'tracking_code' => $tracking_code,
            'submission_id' => $submission_id
        ]);
    } else {
        echo json_encode([
            'success' => true,
            'message' => 'Manuscript submitted successfully! A confirmation email has been sent to your email address.',
            'tracking_code' => $tracking_code,
            'submission_id' => $submission_id
        ]);
    }
    
} catch (Exception $e) {
    error_log('Author Submission Error: ' . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>